/*
 Copyright (C) 2010, Philipp Merkel <linux@philmerk.de>

 Permission to use, copy, modify, and/or distribute this software for any
 purpose with or without fee is hereby granted, provided that the above
 copyright notice and this permission notice appear in all copies.

 THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES WITH
 REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY SPECIAL, DIRECT,
 INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 PERFORMANCE OF THIS SOFTWARE.
 */

#include <linux/input.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/extensions/XTest.h>
#include <X11/keysym.h>
#include <X11/extensions/XInput2.h>
#include "twofingemu.h"
#include "profiles.h"
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#define EXIT_SUCCESS 0
#define EXIT_FAILURE 1

int debugmode = 0;
/* Clickmode: 0 - first finger; 1 - second finger; 2 - center */
int clickmode = 2;

/* Continuation mode -- when 1, two finger gesture is continued when one finger is released. */
#define CONTINUATION 1

#if CONTINUATION
	#define TWO_FINGERS_DOWN fingersDown == 2 && hadTwoFingersOn == 0
	#define TWO_FINGERS_ON fingersDown > 0 && hadTwoFingersOn == 1
	#define TWO_FINGERS_UP fingersDown == 0 && hadTwoFingersOn == 1
#else
	#define TWO_FINGERS_DOWN fingersDown == 2 && fingersWereDown < 2
	#define TWO_FINGERS_ON fingersDown == 2
	#define TWO_FINGERS_UP fingersDown < 2 && fingersWereDown == 2
#endif

/* Daemonize. Source: http://www-theorie.physik.unizh.ch/~dpotter/howto/daemonize (public domain) */
static void daemonize(void) {
	pid_t pid, sid;

	/* already a daemon */
	if (getppid() == 1)
		return;

	/* Fork off the parent process */
	pid = fork();
	if (pid < 0) {
		exit(EXIT_FAILURE);
	}
	/* If we got a good PID, then we can exit the parent process. */
	if (pid > 0) {
		exit(EXIT_SUCCESS);
	}

	/* At this point we are executing as the child process */

	/* Change the file mode mask */
	umask(0);

	/* Create a new SID for the child process */
	sid = setsid();
	if (sid < 0) {
		exit(EXIT_FAILURE);
	}

	/* Change the current working directory.  This prevents the current
	 directory from being locked; hence not being able to remove it. */
	if ((chdir("/")) < 0) {
		exit(EXIT_FAILURE);
	}

	/* Redirect standard files to /dev/null */
	void* r;
	r = freopen("/dev/null", "r", stdin);
	r = freopen("/dev/null", "w", stdout);
	r = freopen("/dev/null", "w", stderr);
}

/* Finger information */
FingerInfo fingerInfos[2] = { { .x=0, .y=0, .down = 0, .id = 0 }, { .x=0, .y=0,
		.down = 0, .id = 1 } };

/* X stuff */
Display* display;
Window ROOT;
int screenNum;
int deviceID;
Atom WM_CLASS;
pthread_t xLoopThread;
Time fairlyCurrentTime;

/* Calibration data */
int calibMinX, calibMaxX, calibMinY, calibMaxY;
unsigned char calibSwapX, calibSwapY, calibSwapAxes;

/* The real width and height of the display, in pixels */
unsigned int realDisplayWidth, realDisplayHeight;
/* The width and height of the (possibly) rotated display as the user perceives it, in pixels */
unsigned int displayWidth, displayHeight;

/* Finger data */
int fingersDown = 0;
int fingersWereDown = 0;
/* Has button press of first button been called in XTest? */
int buttonDown = 0;
/* Shall the mouse pointer be locked at its position? */
int dontMove = 0;
/* Maximum distance that two fingers have been moved while they were on */
double maxDist;
/* The time when the first finger touched */
Time fingerDownTime;
/* Were there once two fingers on during the current touch phase? */
int hadTwoFingersOn = 0;

/* Two finger gestures activated (and input grabbed)? */
int active = 0;
/* Activate two finger gestures (and grab input) as soon as the fingers have been removed */
int activateAtRelease = 0;

/* Gesture information */
int amPerformingGesture = 0;
/* Current profile has scrollBraceAction, thus mouse pointer has to be moved during scrolling */
int dragScrolling = 0;

/* position of center between fingers at start of gesture */
int gestureStartCenterX, gestureStartCenterY;
/* distance between two fingers at start of gesture */
double gestureStartDist;
/* angle of two fingers at start of gesture */
double gestureStartAngle;
/* current position of center between fingers */
int currentCenterX, currentCenterY;

#define PI 3.141592654

/* Profile of current window, if activated */
Profile* currentProfile = NULL;

/* Handle errors by, well, throwing them away. */
int invalidWindowHandler(Display *dsp, XErrorEvent *err) {
	return 0;
}

/* Grab the device so input is captured */
void grab(Display* display, int grabDeviceID) {
	XIEventMask device_mask;
	unsigned char mask_data[1];
	mask_data[0] = 1; /* To prevent warning */
	device_mask.deviceid = grabDeviceID;
	device_mask.mask_len = sizeof(mask_data);
	device_mask.mask = mask_data;
	XISetMask(device_mask.mask, XI_ButtonPress);
	XISetMask(device_mask.mask, XI_ButtonRelease);
	XISetMask(device_mask.mask, XI_Motion);

	XIGrabModifiers modifiers[1] = { { 0, 0 } };

	XIGrabButton(display, grabDeviceID, 1, ROOT, None, GrabModeAsync,
			GrabModeAsync, False, &device_mask, 1, modifiers);

}

/* Ungrab the device so input can be handled by application directly */
void ungrab(Display* display, int grabDeviceID) {
	XIGrabModifiers modifiers[1] = { { 0, 0 } };
	XIUngrabButton(display, grabDeviceID, 1, ROOT, 1, modifiers);

}

/* Activates two finger gesture handling and input grabbing */
void activate() {
	activateAtRelease = 0;
	if (active == 0) {
		grab(display, deviceID);
		active = 1;
	}
}

/* Deactivates two finger gesture handling and input grabbing */
void deactivate() {
	activateAtRelease = 0;
	if (active) {
		active = 0;
		ungrab(display, deviceID);
	}
}

/* Send an XTest event to release the first button if it is currently pressed */
void releaseButton() {
	if (buttonDown) {
		buttonDown = 0;
		XTestFakeButtonEvent(display, 1, False, CurrentTime);
		XFlush(display);
	}
}

/* Executes the given action -- synthesizes key/button press, release or both, depenging
 * on value of whatToDo (EXECUTEACTION_PRESS/_RELEASE/_BOTH). */
void executeAction(Action* action, int whatToDo) {
	if (whatToDo & EXECUTEACTION_PRESS) {
		if (action->actionType != ACTIONTYPE_NONE && action->modifier != 0) {
			if (action->modifier & MODIFIER_SHIFT) {
				XTestFakeKeyEvent(display,
						XKeysymToKeycode(display, XK_Shift_L), True,
						CurrentTime);
			}
			if (action->modifier & MODIFIER_CONTROL) {
				XTestFakeKeyEvent(display, XKeysymToKeycode(display,
						XK_Control_L), True, CurrentTime);
			}
			if (action->modifier & MODIFIER_ALT) {
				XTestFakeKeyEvent(display, XKeysymToKeycode(display, XK_Alt_L),
						True, CurrentTime);
			}
			if (action->modifier & MODIFIER_SUPER) {
				XTestFakeKeyEvent(display,
						XKeysymToKeycode(display, XK_Super_L), True,
						CurrentTime);
			}
		}

		switch (action->actionType) {
		case ACTIONTYPE_BUTTONPRESS:
			XTestFakeButtonEvent(display, action->keyButton, True, CurrentTime);
			break;
		case ACTIONTYPE_KEYPRESS:
			XTestFakeKeyEvent(display, XKeysymToKeycode(display,
					action->keyButton), True, CurrentTime);
			break;
		}

	}

	if (whatToDo & EXECUTEACTION_RELEASE) {

		switch (action->actionType) {
		case ACTIONTYPE_BUTTONPRESS:
			XTestFakeButtonEvent(display, action->keyButton, False, CurrentTime);
			break;
		case ACTIONTYPE_KEYPRESS:
			XTestFakeKeyEvent(display, XKeysymToKeycode(display,
					action->keyButton), False, CurrentTime);
			break;
		}

		if (action->actionType != ACTIONTYPE_NONE && action->modifier != 0) {
			if (action->modifier & MODIFIER_SHIFT) {
				XTestFakeKeyEvent(display,
						XKeysymToKeycode(display, XK_Shift_L), False,
						CurrentTime);
			}
			if (action->modifier & MODIFIER_CONTROL) {
				XTestFakeKeyEvent(display, XKeysymToKeycode(display,
						XK_Control_L), False, CurrentTime);
			}
			if (action->modifier & MODIFIER_ALT) {
				XTestFakeKeyEvent(display, XKeysymToKeycode(display, XK_Alt_L),
						False, CurrentTime);
			}
			if (action->modifier & MODIFIER_SUPER) {
				XTestFakeKeyEvent(display,
						XKeysymToKeycode(display, XK_Super_L), False,
						CurrentTime);
			}
		}
	}

}

/* Returns the active top-level window. A top-level window is one that has WM_CLASS set.
 * May also return None. */
Window getCurrentWindow() {

	/* First get the window that has the input focus */
	Window currentWindow;
	int revert;
	XGetInputFocus(display, &currentWindow, &revert);

	if (currentWindow == None)
		return currentWindow;

	/* Now go through parent windows until we find one with WM_CLASS set. */
	Window root, parent;
	Window* childWindows = NULL;
	unsigned int childCount;


	XClassHint* classHint = XAllocClassHint();

	int i = 0;
	while (1) {
		i++;
		if(i >= 5) return None;
		if (currentWindow == ROOT) {
			/* No top-level window available. Should never happen. */
			XFree(classHint);
			return currentWindow;
		}

		if (XGetClassHint(display, currentWindow, classHint) == 0) {
			XQueryTree(display, currentWindow, &root, &parent, &childWindows,
					&childCount);

			if (childWindows != NULL)
				XFree(childWindows);

			if(currentWindow == parent) {
				/* something wrong */
				return currentWindow;
			}
			currentWindow = parent;
		} else {
			XFree(classHint);
			return currentWindow;
		}
	}
}

/* Returns a pointer to the profile of the currently selected
 * window, or defaultProfile if there is no specific profile for it or the window is invalid. */
Profile* getWindowProfile(Window w) {
	if (w != None) {

		XClassHint* classHint = XAllocClassHint();

		/* Read WM_CLASS member */
		if (XGetClassHint(display, w, classHint)) {

			int i;
			/* Look for the profile with this class */
			for (i = 0; i < profileCount; i++) {
				if (strncmp(classHint->res_name, profiles[i].windowClass, 30)
						== 0) {
					XFree(classHint);
					/* Return this profile */
					return &profiles[i];
				}
			}

			XFree(classHint);
			/* No profile found, return default. */
			return &defaultProfile;
		} else {
			XFree(classHint);
			/* Couldn't get WM_CLASS, return default profile. */
			return &defaultProfile;
		}
	} else {
		/* Invalid window, return default window. */
		return &defaultProfile;
	}

}

/* Process the finger data gathered from the last set of events */
void processFingers() {
	/* If we should activate at release, do it. */
	if (buttonDown == 0 && fingersDown == 0 && activateAtRelease != 0) {
		releaseButton();
		activate();
	}

	if (active == 0)
		return;
	if (TWO_FINGERS_DOWN) {
		/* Second finger touched (and maybe first too) */

		maxDist = 0;

		/* Don't move until we know what gesture we're performing. */
		dontMove = 1;

		/* Memorize that there were two fingers on during touch */
		hadTwoFingersOn = 1;

		/* Get current profile */
		currentProfile = getWindowProfile(getCurrentWindow());

		/* If there had already been a single-touch event raised because the
		 * user was too slow, stop it now. */
		releaseButton();

		/* Calculate center position and distance between touch points */
		gestureStartCenterX = (fingerInfos[0].x + fingerInfos[1].x) / 2;
		gestureStartCenterY = (fingerInfos[0].y + fingerInfos[1].y) / 2;

		int xdiff = fingerInfos[1].x - fingerInfos[0].x;
		int ydiff = fingerInfos[1].y - fingerInfos[0].y;
		gestureStartDist = sqrt(xdiff * xdiff + ydiff * ydiff);
		gestureStartAngle = atan2(ydiff, xdiff) * 180 / PI;

		/* We have not decided on a gesture yet. */
		amPerformingGesture = GESTURE_UNDECIDED;

		/* Move pointer to center between touch points */
		XTestFakeMotionEvent(display, -1, gestureStartCenterX,
				gestureStartCenterY, CurrentTime);

	} else if (TWO_FINGERS_ON) {
		/* Moved with two fingers */

		if(fingersDown == 2) {
			/* Calculate new center between fingers */
			currentCenterX = (fingerInfos[0].x + fingerInfos[1].x) / 2;
			currentCenterY = (fingerInfos[0].y + fingerInfos[1].y) / 2;
		} else {
			int i;
			for(i = 0; i <= 1; i++) {
				if(fingerInfos[i].down) {
					currentCenterX = fingerInfos[i].x;
					currentCenterY = fingerInfos[i].y;
				}
			}
		}

		/* If we are dragScrolling (we are scrolling and there is a brace action,
		 * we need to move the pointer */
		if (amPerformingGesture == GESTURE_SCROLL && dragScrolling) {
			/* Move pointer to center between touch points */
			XTestFakeMotionEvent(display, -1, currentCenterX, currentCenterY,
					CurrentTime);
		}

		/* Perform gestures as long as there are some. */
		while (checkGesture())
			;

	} else if (TWO_FINGERS_UP) {
		/* Second finger (and maybe also first) released */

		/* If there was a scroll gesture and we have a brace action, perform release. */
		if (amPerformingGesture == GESTURE_SCROLL) {
			if (currentProfile->scrollInherit) {
				executeAction(&(defaultProfile.scrollBraceAction),
						EXECUTEACTION_RELEASE);
			} else {
				executeAction(&(currentProfile->scrollBraceAction),
						EXECUTEACTION_RELEASE);
			}
		}
		/* If we haven't performed a gesture and haven't moved too far, perform tap action. */
		if ((amPerformingGesture == GESTURE_NONE || amPerformingGesture
				== GESTURE_UNDECIDED) && maxDist < 10) {
			/* Move pointer to correct position */
			if(clickmode == 2) {
				/* We still are at center */
			} else {
				/* Assume first finger is at ID 0 and second finger at ID 1, might have to be changed later */
				XTestFakeMotionEvent(display, -1, fingerInfos[clickmode].x, fingerInfos[clickmode].y,
						CurrentTime);
			}

			if (currentProfile->tapInherit) {
				executeAction(&(defaultProfile.tapAction), EXECUTEACTION_BOTH);
			} else {
				executeAction(&(currentProfile->tapAction), EXECUTEACTION_BOTH);
			}
		}

		amPerformingGesture = GESTURE_NONE;

	} else if (fingersDown == 1 && fingersWereDown == 0) {
		/* First finger touched */
		fingerDownTime = fairlyCurrentTime;

	} else if (fingersDown == 1) {
		/* Moved with one finger */
		if (hadTwoFingersOn == 0 && buttonDown == 0) {
			if (fairlyCurrentTime > fingerDownTime + CLICK_DELAY) {
				/* Delay has passed, no gesture been performed, so perform single-touch press now */
				buttonDown = 1;
				XTestFakeButtonEvent(display, 1, True, CurrentTime);
				XFlush(display);
			}
		}

	} else if (fingersDown == 0 && fingersWereDown > 0) {
		/* First finger released */
		if (hadTwoFingersOn == 0 && buttonDown == 0) {
			/* The button press time has not been reached yet, and we never had two
			 * fingers on (we could not have done this in this short time) so
			 * we simulate button down and up now. */
			XTestFakeButtonEvent(display, 1, True, CurrentTime);
			XFlush(display);
			XTestFakeButtonEvent(display, 1, False, CurrentTime);
			XFlush(display);
		} else {
			/* We release the button if it is down. */
			releaseButton();
		}
	}
	if (fingersDown == 0) {
		/* Reset fields after release */
		hadTwoFingersOn = 0;
		dontMove = 0;
	}
	/* Save number of fingers to compare next time */
	fingersWereDown = fingersDown;

	/* If we should activate at release, do it (now again because fingersDown might have changed). */
	if (buttonDown == 0 && fingersDown == 0 && activateAtRelease != 0) {
		releaseButton();
		activate();
	}
}

/* All the gesture-related code.
 * Returns 1 if the method should be called again, 0 otherwise.
 */
int checkGesture() {

	/* Calculate difference between two touch points and angle */
	int xdiff = fingerInfos[1].x - fingerInfos[0].x;
	int ydiff = fingerInfos[1].y - fingerInfos[0].y;
	double currentDist = sqrt(xdiff * xdiff + ydiff * ydiff);
	double currentAngle = atan2(ydiff, xdiff) * 180 / PI;

	/* Check distance the fingers (more exactly: the point between them)
	 * has been moved since start of the gesture. */
	int xdist = currentCenterX - gestureStartCenterX;
	int ydist = currentCenterY - gestureStartCenterY;
	double moveDist = sqrt(xdist * xdist + ydist * ydist);
	if (moveDist > maxDist && fingersDown == 2)
		maxDist = moveDist;

	/* We don't know yet what to do, so look if we can decide now. */
	if (amPerformingGesture == GESTURE_UNDECIDED && fingersDown == 2) {
		int scrollMinDist = currentProfile->scrollMinDistance;
		if (currentProfile->scrollInherit)
			scrollMinDist = defaultProfile.scrollMinDistance;
		if ((int) moveDist > scrollMinDist) {
			amPerformingGesture = GESTURE_SCROLL;

			if (currentProfile->scrollInherit) {
				executeAction(&(defaultProfile.scrollBraceAction),
						EXECUTEACTION_PRESS);
				if (defaultProfile.scrollBraceAction.actionType
						== ACTIONTYPE_NONE) {
					dragScrolling = 0;
				} else {
					dragScrolling = 1;
				}
			} else {
				executeAction(&(currentProfile->scrollBraceAction),
						EXECUTEACTION_PRESS);
				if (currentProfile->scrollBraceAction.actionType
						== ACTIONTYPE_NONE) {
					dragScrolling = 0;
				} else {
					dragScrolling = 1;
				}
			}
			return 1;
		}

		int zoomMinDist = currentProfile->zoomMinDistance;
		if (currentProfile->zoomInherit)
			zoomMinDist = defaultProfile.zoomMinDistance;
		if (abs((int) currentDist - gestureStartDist) > zoomMinDist) {
			amPerformingGesture = GESTURE_ZOOM;
			return 1;
		}

		int rotateMinDist = currentProfile->rotateMinDistance;
		double rotateMinAngle = currentProfile->rotateMinAngle;
		if (currentProfile->rotateInherit) {
			rotateMinDist = defaultProfile.rotateMinDistance;
			rotateMinAngle = defaultProfile.rotateMinAngle;
		}
		double rotatedBy = currentAngle - gestureStartAngle;
		if (rotatedBy < -180)
			rotatedBy += 360;
		if (rotatedBy > 180)
			rotatedBy -= 360;
		//printf("Rotated by: %f; min. angle: %f\n", rotatedBy, rotateMinAngle);
		if (abs(rotatedBy) > rotateMinAngle && (int) currentDist
				> rotateMinDist) {
			amPerformingGesture = GESTURE_ROTATE;
			return 1;
		}
	}

	/* If we know hat gesture to perform, look if there is something to do */
	switch (amPerformingGesture) {
	case GESTURE_SCROLL:
		;
		int hscrolledBy = currentCenterX - gestureStartCenterX;
		int vscrolledBy = currentCenterY - gestureStartCenterY;
		int hscrollStep = currentProfile->hscrollStep;
		int vscrollStep = currentProfile->vscrollStep;
		if (currentProfile->scrollInherit) {
			hscrollStep = defaultProfile.hscrollStep;
			vscrollStep = defaultProfile.vscrollStep;
		}
		if (hscrollStep == 0 || vscrollStep == 0)
			return 0;

		if (hscrolledBy > hscrollStep) {
			if (currentProfile->scrollInherit) {
				executeAction(&(defaultProfile.scrollRightAction),
						EXECUTEACTION_BOTH);
			} else {
				executeAction(&(currentProfile->scrollRightAction),
						EXECUTEACTION_BOTH);
			}

			gestureStartCenterX = gestureStartCenterX + hscrollStep;
			return 1;
		}
		if (hscrolledBy < -hscrollStep) {
			if (currentProfile->scrollInherit) {
				executeAction(&(defaultProfile.scrollLeftAction),
						EXECUTEACTION_BOTH);
			} else {
				executeAction(&(currentProfile->scrollLeftAction),
						EXECUTEACTION_BOTH);
			}

			gestureStartCenterX = gestureStartCenterX - hscrollStep;
			return 1;
		}
		if (vscrolledBy > vscrollStep) {
			if (currentProfile->scrollInherit) {
				executeAction(&(defaultProfile.scrollDownAction),
						EXECUTEACTION_BOTH);
			} else {
				executeAction(&(currentProfile->scrollDownAction),
						EXECUTEACTION_BOTH);
			}

			gestureStartCenterY = gestureStartCenterY + vscrollStep;
			return 1;
		}
		if (vscrolledBy < -vscrollStep) {
			if (currentProfile->scrollInherit) {
				executeAction(&(defaultProfile.scrollUpAction),
						EXECUTEACTION_BOTH);
			} else {
				executeAction(&(currentProfile->scrollUpAction),
						EXECUTEACTION_BOTH);
			}

			gestureStartCenterY = gestureStartCenterY - vscrollStep;
			return 1;
		}

		return 0;
	case GESTURE_ZOOM:
		;
		double zoomedBy = currentDist / gestureStartDist;
		double zoomStep = currentProfile->zoomStep;
		if (currentProfile->zoomInherit)
			zoomStep = defaultProfile.zoomStep;
		if (zoomedBy > zoomStep) {
			if (currentProfile->zoomInherit) {
				executeAction(&(defaultProfile.zoomInAction),
						EXECUTEACTION_BOTH);
			} else {
				executeAction(&(currentProfile->zoomInAction),
						EXECUTEACTION_BOTH);
			}
			/* Reset distance */
			gestureStartDist = gestureStartDist * zoomStep;
			return 1;
		} else if (zoomedBy < 1 / zoomStep) {
			if (currentProfile->zoomInherit) {
				executeAction(&(defaultProfile.zoomOutAction),
						EXECUTEACTION_BOTH);
			} else {
				executeAction(&(currentProfile->zoomOutAction),
						EXECUTEACTION_BOTH);
			}
			/* Reset distance */
			gestureStartDist = gestureStartDist / zoomStep;
			return 1;
		}
		return 0;
	case GESTURE_ROTATE:
		;
		double rotatedBy = currentAngle - gestureStartAngle;
		if (rotatedBy < -180)
			rotatedBy += 360;
		if (rotatedBy > 180)
			rotatedBy -= 360;
		double rotateStep = currentProfile->rotateStep;
		if (currentProfile->rotateInherit)
			rotateStep = defaultProfile.rotateStep;
		if (rotatedBy > rotateStep) {
			if (currentProfile->rotateInherit) {
				executeAction(&(defaultProfile.rotateRightAction),
						EXECUTEACTION_BOTH);
			} else {
				executeAction(&(currentProfile->rotateRightAction),
						EXECUTEACTION_BOTH);
			}

			gestureStartAngle = gestureStartAngle + rotateStep;
		} else if (rotatedBy < -rotateStep) {
			if (currentProfile->rotateInherit) {
				executeAction(&(defaultProfile.rotateLeftAction),
						EXECUTEACTION_BOTH);
			} else {
				executeAction(&(currentProfile->rotateLeftAction),
						EXECUTEACTION_BOTH);
			}

			gestureStartAngle = gestureStartAngle - rotateStep;
		}

		return 0;
	}

	return 0;

}

/* Called when a blacklisted window is left (for some reasons sometimes also if it's not blacklisted) */
void leaveWindow() {
	if (active == 0) {

		activateAtRelease = 1;
	}
}

/* Called when a blacklisted window is focussed */
void enterBlacklistedWindow() {
	//printf("Entered bad window.\n");
	activateAtRelease = 0;
	releaseButton();
	deactivate();
}

/* Returns whether the given window is blacklisted */
int isWindowBlacklisted(Window w) {
	int i;

	XClassHint* classHint = XAllocClassHint();

	if (XGetClassHint(display, w, classHint)) {
		printf("Found window with id %i and class '%s' \r\n", (int) w,
				classHint->res_name);

		for (i = 0; i < blacklistLength; i++) {
			if (strncmp(classHint->res_name, blacklist[i], 30) == 0) {
				XFree(classHint);
				return 1;
			}
		}

	}
	XFree(classHint);
	return 0;
}

/* Called when a new window is mapped. Checks whether it is blacklisted and registers
 * the activation events if yes. Then checks if it is active and, if yes, calls enter/leave.
 */
void windowMapped(Window w) {

	if (isWindowBlacklisted(w)) {
		/* register for window focus events */
		XSelectInput(display, w, EnterWindowMask | LeaveWindowMask);
		if (getCurrentWindow() == w) {
			/* If this is current and blacklisted, call enterBlacklistedWindow */
			enterBlacklistedWindow();
		}
	} else {
		if (getCurrentWindow() == w) {
			/* It is current and not blacklisted, so we might have left a blacklisted window */
			leaveWindow();
		}
	}

}

/* X thread that processes X events in parallel to kernel device loop */
void* xLoopThreadFunction(void* arg) {
	XEvent ev;

	while (1) {
		XNextEvent(display, &ev);

		if (XGetEventData(display, (XGenericEventCookie *) &ev)) {
			XGenericEventCookie *cookie = &ev.xcookie;
			if (cookie->evtype == XI_Motion) {
				/* Move event */
				XIDeviceEvent* data = cookie->data;
				fairlyCurrentTime = data->time;
				if (!dontMove) {
					/* Fake single-touch move event */
					XTestFakeMotionEvent(display, -1, (int) data->root_x,
							(int) data->root_y, CurrentTime);
				}
			} else if (cookie->evtype == XI_PropertyEvent) {
				/* Device properties changed -> recalibrate. */
				printf("Device properties changed.\n");
				readCalibrationData(0);
			}

			XFreeEventData(display, (XGenericEventCookie *) &ev);

		} else {
			if (ev.type == MapNotify) {
				windowMapped(ev.xmap.window);
			} else if (ev.type == EnterNotify) {
				/* This is only called for blacklisted windows */
				enterBlacklistedWindow();
			} else if (ev.type == LeaveNotify) {
				/* This should only be called for blacklisted windows, but it's sometimes also
				 * called when a non-blacklisted window is left. But that's no problem for us. */
				leaveWindow();
				//			} else if(ev.type == ConfigureNotify) {
				//				displayWidth = DisplayWidth(display, screenNum);
				//				displayHeight = DisplayHeight(display, screenNum);
				//				printf("Display width: %i; height: %i\n", displayWidth, displayHeight);

			}
		}
	}
}

/* Checks for all running windows if they are blacklisted and/or the current window.
 * Registers enter/leave event handlers for blacklisted windows. Executes them for
 * the current window. */
void checkRunningWindows() {
	Window root;
	Window parent;
	Window* childWindows = NULL;
	unsigned int childCount;
	/* Get all top-level windows */
	XQueryTree(display, ROOT, &root, &parent, &childWindows, &childCount);

	if (childWindows != NULL) {
		int i;
		for (i = 0; i < childCount; i++) {
			windowMapped(childWindows[i]);
		}
		XFree(childWindows);
	}
}

/* Reads the calibration data from evdev, should be self-explanatory. */
void readCalibrationData(int exitOnFail) {

	Atom retType;
	int retFormat;
	unsigned long retItems, retBytesAfter;
	unsigned int* data;
	XIGetProperty(display, deviceID, XInternAtom(display,
			"Evdev Axis Calibration", 0), 0, 4 * 32, False, XA_INTEGER,
			&retType, &retFormat, &retItems, &retBytesAfter,
			(unsigned char**) &data);

	if (retItems != 4 || data[0] == data[1] || data[2] == data[3]) {
		/* evdev might not be ready yet after resume. Let's wait a second and try again. */
		sleep(1);

		XIGetProperty(display, deviceID, XInternAtom(display,
				"Evdev Axis Calibration", 0), 0, 4 * 32, False, XA_INTEGER,
				&retType, &retFormat, &retItems, &retBytesAfter,
				(unsigned char**) &data);

		if (retItems != 4 || data[0] == data[1] || data[2] == data[3]) {

			if (exitOnFail) {
				printf(
						"No valid calibration data found. Evdev Axis Calibration not set?\n");
				exit(1);
			} else {
				return;
			}
		}
	}
	calibMinX = data[0];
	calibMaxX = data[1];
	calibMinY = data[2];
	calibMaxY = data[3];

	XFree(data);

	unsigned char* data2;

	XIGetProperty(display, deviceID, XInternAtom(display,
			"Evdev Axis Inversion", 0), 0, 2 * 8, False, XA_INTEGER, &retType,
			&retFormat, &retItems, &retBytesAfter, (unsigned char**) &data2);

	if (retItems != 2) {
		if (exitOnFail) {
			printf("No valid axis inversion data found.\n");
			exit(1);
		} else {
			return;
		}
	}

	calibSwapX = data2[0];
	calibSwapY = data2[1];

	XFree(data2);

	XIGetProperty(display, deviceID,
			XInternAtom(display, "Evdev Axes Swap", 0), 0, 8, False,
			XA_INTEGER, &retType, &retFormat, &retItems, &retBytesAfter,
			(unsigned char**) &data2);

	if (retItems != 1) {
		if (exitOnFail) {
			printf("No valid axes swap data found.\n");
			exit(1);
		} else {
			return;
		}
	}
	calibSwapAxes = data2[0];
	if (calibSwapAxes) {
		displayWidth = realDisplayHeight;
		displayHeight = realDisplayWidth;
	} else {
		displayWidth = realDisplayWidth;
		displayHeight = realDisplayHeight;
	}

	XFree(data2);

}

/* Main function, contains kernel driver event loop */
int main(int argc, char **argv) {

	char* devname = 0;
	int doDaemonize = 1;
	int doWait = 0;

	int i;
	for (i = 1; i < argc; i++) {
		if (strcmp(argv[i], "--debug") == 0) {
			doDaemonize = 0;
			debugmode = 1;
		} else if (strcmp(argv[i], "--wait") == 0) {
			doWait = 1;
		} else if (strcmp(argv[i], "--click=first") == 0) {
			clickmode = 0;
		} else if (strcmp(argv[i], "--click=second") == 0) {
			clickmode = 1;
		} else if (strcmp(argv[i], "--click=center") == 0) {
			clickmode = 2;
		} else {
			devname = argv[i];
		}

	}



	if (doDaemonize) {
		daemonize();
	}

	if (doWait) {
		/* Wait until all necessary things are loaded */
		sleep(10);
	}

	/* We have two threads accessing X */
	XInitThreads();

	/* Connect to X server */
	if ((display = XOpenDisplay(NULL)) == NULL) {
		fprintf(stderr, "Couldn't connect to X server\n");
		exit(1);
	}

	/* Read X data */
	screenNum = DefaultScreen(display);

	ROOT = RootWindow(display, screenNum);

	realDisplayWidth = DisplayWidth(display, screenNum);
	realDisplayHeight = DisplayHeight(display, screenNum);

	WM_CLASS = XInternAtom(display, "WM_CLASS", 0);

	/* Get notified about new windows */
	XSelectInput(display, ROOT, StructureNotifyMask | SubstructureNotifyMask);

	//TODO load blacklist and profiles from file(s)

	/* Device file name */
	if (devname == 0) {
		devname = "/dev/twofingtouch";
	}

	/* Save return value of pthread_create so we only call it later if it hasn't been successfully called before. */
	int threadReturn = 1;

	/* Try to read from device file */
	int fileDesc;
	if ((fileDesc = open(devname, O_RDONLY)) < 0) {
		perror("twofing");
		return 1;
	}

	while (1) {
		/* Perform initialization at beginning and after module has been re-loaded */
		int rd, i;
		struct input_event ev[64];

		char name[256] = "Unknown";

		/* Read device name */
		ioctl(fileDesc, EVIOCGNAME(sizeof(name)), name);
		printf("Input device name: \"%s\"\n", name);

		XSetErrorHandler(invalidWindowHandler);

		/* XInput Extension available? */
		int opcode, event, error;
		if (!XQueryExtension(display, "XInputExtension", &opcode, &event,
				&error)) {
			printf("X Input extension not available.\n");
			exit(1);
		}

		/* Which version of XI2? We support 2.0 */
		int major = 2, minor = 0;
		if (XIQueryVersion(display, &major, &minor) == BadRequest) {
			printf("XI2 not available. Server supports %d.%d\n", major, minor);
			exit(1);
		}

		int n;
		XIDeviceInfo *info = XIQueryDevice(display, XIAllDevices, &n);
		if (!info) {
			printf("No XInput devices available\n");
			exit(1);
		}

		/* Go through input devices and look for that with the same name as the given device */
		int devindex;
		for (devindex = 0; devindex < n; devindex++) {
			if (info[devindex].use == XIMasterPointer || info[devindex].use
					== XIMasterKeyboard)
				continue;

			if (strcmp(info[devindex].name, name) == 0) {
				deviceID = info[devindex].deviceid;

				break;
			}

		}
		if (deviceID == -1) {
			printf("Input device not found in XInput device list.\n");
			exit(1);
		}

		XIFreeDeviceInfo(info);

		printf("XInput device id is %i.\n", deviceID);

		/* Prepare by reading calibration */
		readCalibrationData(1);

		/* Receive device property change events */
		XIEventMask device_mask2;
		unsigned char mask_data2[2] = { 0, 0 };
		device_mask2.deviceid = deviceID;
		device_mask2.mask_len = sizeof(mask_data2);
		device_mask2.mask = mask_data2;
		XISetMask(device_mask2.mask, XI_PropertyEvent);
		XISelectEvents(display, ROOT, &device_mask2, 1);

		/* Needed for XTest to work correctly */
		XTestGrabControl(display, True);
		active = 0;
		checkRunningWindows();
		/* Don't wait until release but activate right now */
		if (activateAtRelease) {
			activateAtRelease = 0;
			activate();
		}

		/* Needed for some reason to receive events */
		XGrabPointer(display, ROOT, False, 0, GrabModeAsync, GrabModeAsync,
				None, None, CurrentTime);
		XUngrabPointer(display, CurrentTime);

		/* If it is not already running, create thread that listens to XInput events */
		if (threadReturn != 0)
			threadReturn = pthread_create(&xLoopThread, NULL,
					xLoopThreadFunction, NULL);

		printf("Reading input from device ... (interrupt to exit)\n");
		FingerInfo tempFingerInfo = { -1, -1, -1, -1 };

		/* Kernel device event loop */
		while (1) {
			rd = read(fileDesc, ev, sizeof(struct input_event) * 64);
			if (rd < (int) sizeof(struct input_event)) {
				printf("Data stream stopped\n");
				break;
			}
			for (i = 0; i < rd / sizeof(struct input_event); i++) {
				if (ev[i].type == EV_SYN) {

					if (2 == ev[i].code) { // MT_Sync : Multitouch event end
						/* Finger info for one finger collected in tempFingerInfo, so save it to fingerInfos. */
						/* Currently, we assume that the tracking ids of the fingers are 0,1 -- we should not do this but rather go through all fingers and check for the tracking id. We should also check if there are more than two fingers and set all to down then. */
						int index = tempFingerInfo.id;
						if (index == 0 || index == 1) {
							fingerInfos[index].id = tempFingerInfo.id;

							/* Calibrate coordinate */
							if (calibSwapAxes) {
								fingerInfos[index].x = ((tempFingerInfo.y
										- calibMinY) * displayWidth)
										/ (calibMaxY - calibMinY);
								fingerInfos[index].y = ((tempFingerInfo.x
										- calibMinX) * displayHeight)
										/ (calibMaxX - calibMinX);
							} else {

								fingerInfos[index].x = ((tempFingerInfo.x
										- calibMinX) * displayWidth)
										/ (calibMaxX - calibMinX);
								fingerInfos[index].y = ((tempFingerInfo.y
										- calibMinY) * displayHeight)
										/ (calibMaxY - calibMinY);
							}
							if (calibSwapX)
								fingerInfos[index].x = displayWidth
										- fingerInfos[index].x;
							if (calibSwapY)
								fingerInfos[index].y = displayHeight
										- fingerInfos[index].y;

							if (fingerInfos[index].x < 0)
								fingerInfos[index].x = 0;
							if (fingerInfos[index].y < 0)
								fingerInfos[index].y = 0;
							if (fingerInfos[index].x > displayWidth)
								fingerInfos[index].x = displayWidth;
							if (fingerInfos[index].y > displayHeight)
								fingerInfos[index].y = displayHeight;
						}
						fingerInfos[index].down = 1;
					} else if (0 == ev[i].code) { // Ev_Sync event end
						/* All finger data received, so process now. */
						processFingers();

						/* Reset for next set of events */
						fingerInfos[0].down = 0;
						fingerInfos[1].down = 0;
						fingersDown = 0;
						tempFingerInfo.id = -1;
					}
				} else if (ev[i].type == EV_MSC && (ev[i].code == MSC_RAW
						|| ev[i].code == MSC_SCAN)) {
				} else {
					/* Set finger info values for current finger */
					if (ev[i].code == 57) {
						tempFingerInfo.id = ev[i].value;
					};
					if (ev[i].code == 53) {
						tempFingerInfo.x = ev[i].value;
						fingersDown++;
					};
					if (ev[i].code == 54) {
						tempFingerInfo.y = ev[i].value;
					};
				}
			}
		}

		/* Stream stopped, probably because module has been unloaded */
		close(fileDesc);

		/* Clean up */
		if (active) {
			ungrab(display, deviceID);
		}
		releaseButton();

		/* Wait until device file is there again */
		while ((fileDesc = open(devname, O_RDONLY)) < 0) {
			sleep(1);
		}

	}

}

